<?php


namespace YiluTech\FileCenter;

use Illuminate\Support\Str;

trait FilenameTrait
{
    protected string $root = '';

    protected string $prefix = '';

    public function setRoot(?string $path)
    {
        $this->root = empty($path) ? '' : (rtrim($path, '\\/') . '/');
    }

    public function getRoot(): string
    {
        return $this->root;
    }

    public function setPrefix(?string $path)
    {
        $this->prefix = empty($path) ? '' : (rtrim($path, '\\/') . '/');
    }

    public function getPrefix(): string
    {
        return $this->prefix;
    }

    public function getMoveToPath(string $from, ?string $to): string
    {
        if (empty($to)) {
            return basename($from);
        }
        if (substr($to, -1) === '/') {
            return $to . basename($from);
        }
        return $to;
    }

    protected function makeFileName($file, $suffix = null): string
    {
        return $this->timeToStr() . Str::random(16) . '.' . ($suffix ?? $file->guessExtension());
    }

    protected function makeRecyclePath(string $path): string
    {
        return $this->getTempDir() . '/' . $this->encodePath($path);
    }

    protected function getRecyclePath(string $filename): string
    {
        $parts = explode('#', $filename);
        $timestamp = $this->strToTime(current($parts));
        return '.temp/' . date('Ymd', $timestamp) . '/' . $filename;
    }

    protected function getRecoverPath(string $filename): string
    {
        $parts = explode('#', $filename);
        array_shift($parts);
        return implode('/', $parts);
    }

    protected function getTempDir($timestamp = null): string
    {
        return '.temp/' . date('Ymd', $timestamp ?? time());
    }

    protected function encodePath(string $path): string
    {
        return $this->timeToStr() . '#' . str_replace('/', '#', $this->applyPrefix($path, false));
    }

    protected function timeToStr($timestamp = null): string
    {
        return base_convert(floor($timestamp ?? time() / 86400), 10, 36);
    }

    protected function strToTime(string $str): int
    {
        return base_convert($str, 36, 10) * 86400;
    }

    protected function applyPrefix(string $path, $withRoot = true): string
    {
        $path = ltrim($path, '\\/');
        if ($path && $path[0] !== '.') {
            $path = $this->prefix . $path;
        }
        if ($withRoot) {
            $path = $this->applyRoot($path);
        }
        return $path;
    }

    protected function applyRoot(string $path): string
    {
        return $this->root . ltrim($path, '\\/');
    }

    protected function removeRoot($path): string
    {
        return substr($path, strlen($this->root));
    }
}
