<?php

namespace YiluTech\FileCenter;

use Illuminate\Contracts\Database\Eloquent\CastsAttributes;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Model;


class FilePathAttribute implements CastsAttributes
{
    public static array $EXCEPT_PREFIXES = [
        'icon/'
    ];

    protected $dir;

    protected $bucket;

    public function __construct($dir = '', $bucket = null)
    {
        $this->dir = $dir . '/';
        $this->bucket = is_object($bucket) ? $bucket : Facade\FileCenter::bucket($bucket);
    }

    public static function dir($model, string $key, $dir, ...$params)
    {
        $instance = new static('', Facade\FileCenter::dir($dir, ...$params));
        return Attribute::make(
            function ($value, array $attributes) use ($model, $key, $instance) {
                return $instance->get($model, $key, $value, $attributes);
            },
            function ($value, array $attributes) use ($model, $key, $instance) {
                return $instance->set($model, $key, $value, $attributes);
            }
        );
    }

    public function get($model, string $key, $value, array $attributes)
    {
        if (empty($value)) return $value;

        if (self::isJsonStr($value)) {
            $value = json_decode($value, true);
            array_walk($value, function (&$item) {
                if (is_string($item) && !static::isFullUrl($item)) {
                    $item = $this->bucket->getUrl($item);
                }
            });
        } else if (!static::isFullUrl($value)) {
            $value = $this->bucket->getUrl($value);
        }
        return $value;
    }

    public function set($model, string $key, $value, array $attributes)
    {
        if (is_array($value)) {
            array_walk($value, function (&$item) {
                if (is_string($item) && static::movable($item)) {
                    $item = $this->bucket->move($item, $this->dir);
                }
            });
            return [$key => json_encode($value)];
        } else if (is_string($value) && static::movable($value)) {
            $value = $this->bucket->move($value, $this->dir);
        }
        $this->removeOriginal($model, $key, $value);
        return $value;
    }

    protected function removeOriginal(Model $model, $key, $current)
    {
        if ($model->exists && ($original = $model->getRawOriginal($key))) {
            if (is_array($current)) {
                $this->remove(array_diff(json_decode($original), $current));
            } else if ($current != $original) {
                $this->remove($original);
            }
        }
    }

    public static function isJsonStr($str): bool
    {
        return str_starts_with($str, '{') || str_starts_with($str, '[');
    }

    public static function isFullUrl($path): bool
    {
        return str_starts_with($path, 'http://') || str_starts_with($path, 'https://');
    }

    public static function movable(string $path): bool
    {
        if (static::isFullUrl($path)) return false;

        foreach (static::$EXCEPT_PREFIXES as $PREFIX) {
            if (str_starts_with($path, $PREFIX)) return false;
        }
        return true;
    }

    protected function remove($path)
    {
        if (empty($path)) return;

        if (is_array($path)) {
            $path = array_filter($path, [static::class, 'movable']);
        } else if (!static::movable($path)) {
            $path = null;
        }

        if (!empty($path)) {
            $this->bucket->delete($path);
        }
    }
}
