<?php

namespace YiluTech\FileCenter;

use Illuminate\Container\Container;
use Illuminate\Support\Facades\DB;
use YiluTech\FileCenter\Bridge\BridgeInterface;
use YiluTech\FileCenter\Mock\ServerMocker;

class ClientManager
{
    /**
     * @var Client[]
     */
    protected array $buckets = array();

    protected bool $beginBatched = false;

    protected Client $current;

    protected $app;

    public function __construct(Container $app)
    {
        $this->app = $app;
    }

    public function mock(ServerMocker $mocker)
    {
        app()->singleton(BridgeInterface::class, function () use ($mocker) {
            return $mocker->getBridge();
        });
        return $this;
    }

    public function bucket($bucket = null)
    {
        if ($bucket === null) {
            $bucket = env('FILE_CENTER_BUCKET');
        }

        if (empty($bucket)) {
            throw new FileCenterException('Undefined bucket.');
        }

        if (empty($this->buckets[$bucket])) {
            $client = $this->app->make(Client::class, compact('bucket'));
            if ($this->beginBatched) {
                $client->beginBatch();
            }
            $this->buckets[$bucket] = $client;
        }

        return $this->current = $this->buckets[$bucket];
    }

    public function dir($name, ...$params)
    {
        $options = $this->app->config->get('filesystems.dir.' . $name);
        if (!$options) {
            throw new FileCenterException("Undefined filesystem dir \"$name\".");
        }
        $bucket = $this->bucket($options['bucket']);
        if (isset($options['root'])) {
            $prefix = preg_replace_callback('/:(\w+)/', function ($match) use (&$params) {
                $value = array_shift($params);
                if (is_null($value) || is_bool($value)) {
                    throw new FileCenterException("Undefined filesystem dir root params \"$match[1]\".");
                }
                return $value;
            }, $options['root']);
            $bucket->setPrefix($prefix);
        }
        return $bucket;
    }

    public function moveFromHtml(string $content, string $dir = '/', ?string $original = null, $tag = 'img'): string
    {
        $url = preg_quote($this->getUrl(), '/');
        $pattern = "/(?:<($tag)).*?(src=\"$url([\\w\\-\\/#@%.]+)\").*?>/";

        $urls = [];
        $content = preg_replace_callback($pattern, function ($matches) use ($dir, &$urls) {
            if (strncmp($matches[3], '.temp/', 6) === 0) {
                $path = $this->move($matches[3], $dir);
                $attr = str_replace($matches[3], $path, $matches[2]);
                $matches[0] = str_replace($matches[2], $attr, $matches[0]);
                $matches[3] = $path;
            }
            $urls[] = $matches[3];
            return $matches[0];
        }, $content);

        if (isset($original) && preg_match_all($pattern, $original, $matches)) {
            $diff = array_diff($matches[3], $urls);
            if (!empty($diff)) {
                $this->delete($diff);
            }
        }
        return $content;
    }

    public function withDBTransition($callback)
    {
        try {
            $this->beginBatch();
            DB::beginTransaction();
            $result = $callback();
            $this->commit();
            DB::commit();
        } catch (\Exception $exception) {
            DB::rollBack();
            $this->rollback();
            throw $exception;
        }
        return $result;
    }

    public function beginBatch()
    {
        foreach ($this->buckets as $bucket) {
            $bucket->beginBatch();
        }
        $this->beginBatched = true;
    }

    public function commit()
    {
        foreach ($this->buckets as $bucket) {
            if ($bucket->isBeginBatched()) $bucket->commit();
        }
        $this->beginBatched = false;
    }

    public function rollback()
    {
        foreach ($this->buckets as $bucket) {
            $bucket->rollback();
        }
    }

    public function __call($name, $arguments)
    {
        if (empty($this->current)) {
            $this->current = $this->bucket();
        }
        return $this->current->{$name}(...$arguments);
    }
}
