<?php

namespace YiluTech\FileCenter;

use YiluTech\FileCenter\Bridge\BridgeInterface;

class Client
{
    use FilenameTrait;

    protected array $files = [];

    protected array $committed = [];

    protected bool $isBatched = false;

    protected string $bucket;

    protected string $urlPrefix;

    protected BridgeInterface $bridge;

    public function __construct(BridgeInterface $bridge, string $bucket)
    {
        $this->bridge = $bridge;
        $this->bucket = $bucket;
    }

    public function getBucket(): string
    {
        return $this->bucket;
    }

    public function getUrl($path = ''): string
    {
        if (empty($this->urlPrefix)) {
            $this->urlPrefix = $this->exec('getUrl');
        }
        return $this->urlPrefix . $this->applyPrefix($path);
    }

    public function exists(string $path)
    {
        return $this->exec('exists', $this->applyPrefix($path));
    }

    public function filter(array $paths)
    {
        return $this->exec('filter', array_map([$this, 'applyPrefix'], $paths));
    }

    public function move(string $from, string $to = '/')
    {
        $to = $this->getMoveToPath($from, $to);
        $from = $this->applyPrefix($from);
        $target = $this->applyPrefix($to);
        if ($from === $target) return $to;
        if ($this->isBatched) {
            $this->files[$target] = $this->files[$from] ?? $from;
            unset($this->files[$from]);
        } else {
            if (!$this->exec('move', $from, $target)) {
                throw new FileCenterException(sprintf('move file %s error', $from));
            }
        }
        return $to;
    }

    public function moveMany(array $from, string $to = '/')
    {
        if (empty($from)) return $from;
        $isBatched = $this->isBatched;
        $this->beginBatch();
        foreach ($from as $index => $path) {
            $from[$index] = $this->move($path, is_array($to) ? $to[$index] : $to);
        }
        if (!$isBatched) {
            $this->commit();
        }
        return $from;
    }

    /**
     * 删除文件
     *
     * @param $path string|array
     * @return bool
     */
    public function delete($path)
    {
        $paths = is_array($path) ? $path : func_get_args();

        if (!count($paths)) return true;

        if ($this->isBatched) {
            foreach ($paths as $path) {
                $path = $this->applyPrefix($path);
                $this->files[$this->makeRecyclePath($path)] = $path;
            }
        } else {
            $this->exec('delete', $paths);
        }
        return true;
    }

    public function beginBatch()
    {
        if (!$this->isBatched) {
            $this->isBatched = true;
            $this->files = [];
        }
    }

    public function isBeginBatched(): bool
    {
        return $this->isBatched;
    }

    public function commit()
    {
        $bool = true;
        if (!empty($this->files)) {
            $bool = $this->exec('moveMany', array_values($this->files), array_keys($this->files));
            $this->committed = $this->files;
            $this->files = [];
        }
        $this->isBatched = false;
        return $bool;
    }

    /**
     * 回滚操作记录
     *
     */
    public function rollback()
    {
        if ($this->isBatched) {
            $this->isBatched = false;
        }
        $bool = true;
        if (!empty($this->committed)) {
            $bool = $this->exec('moveMany', array_keys($this->committed), array_values($this->committed));
            $this->committed = [];
        }
        return $bool;
    }

    protected function exec($action, ...$params)
    {
        return $this->bridge->call($this->bucket, $action, $params);
    }
}
