<?php

namespace YiluTech\FileCenter\Adapter;

use League\Flysystem\AdapterInterface;
use League\Flysystem\Config;
use League\Flysystem\FileExistsException;

class Memory implements AdapterInterface
{
    private Dir $root;

    public function __construct()
    {
        $this->root = new Dir('');
    }

    /**
     * @param string $path
     * @param string|null $name
     * @return array|string
     */
    public function pathInfo(string $path, string $name = null)
    {
        $info = pathinfo($path);
        $info['path'] = isset($info['dirname']) && $info['dirname'] !== '\\' ? preg_split('/[\\\\\/]/', $info['dirname']) : [];
        if (!empty($info['basename'])) {
            $info['path'][] = $info['basename'];
        }
        return $name === null ? $info : $info[$name] ?? null;
    }

    protected function get(DirInterface $dir, array $path)
    {
        foreach ($path as $item) {
            if ($dir instanceof DirInterface === false) {
                return null;
            }
            $dir = $dir->getFile($item);
        }
        return $dir;
    }

    protected function makeDir(array $path, ?DirInterface $dir = null)
    {
        foreach ($path as $name) {
            if (!$dir || !($_ = $dir->getFile($name))) {
                $_ = new Dir($name);
                $_->attach($dir);
            }
            $dir = $_;
        }
        return $dir;
    }

    protected function makeFile(array $path, ?DirInterface $dir = null): FileInterface
    {
        $filename = array_pop($path);
        $dir = $this->makeDir($path, $dir);
        $file = $dir->getFile($filename);
        if ($file && $file instanceof FileInterface === false) {
            throw new FileExistsException($file->getFilePath());
        }
        if (!$file) {
            $file = new File($filename);
            $file->attach($dir);
        }
        return $file;
    }

    public function has($path): bool
    {
        return $this->get($this->root, $this->pathInfo($path, 'path')) instanceof FileInterface;
    }

    public function read($path)
    {
        $file = $this->get($this->root, $this->pathInfo($path, 'path'));
        if ($file instanceof FileInterface) {
            return ['type' => $file->getType(), 'path' => $file->getFilePath(), 'contents' => $file->getContent()];
        }
        return false;
    }

    public function readStream($path)
    {
        $file = $this->get($this->root, $this->pathInfo($path, 'path'));
        if ($file instanceof FileInterface) {
            return ['type' => $file->getType(), 'path' => $file->getFilePath(), 'stream' => $file->getStream()];
        }
        return false;
    }

    public function getVisibility($path)
    {
        $target = $this->get($this->root, $this->pathInfo($path, 'path'));
        if (!$target) {
            return false;
        }
        return ['path' => $target->getFilePath(), 'visibility' => $target->getVisibility()];
    }

    public function setVisibility($path, $visibility)
    {
        $target = $this->get($this->root, $this->pathInfo($path, 'path'));
        if (!$target) {
            return false;
        }
        $target->setVisibility($visibility);
        return ['path' => $target->getFilePath(), 'visibility' => $target->getVisibility()];
    }

    public function delete($path)
    {
        $target = $this->get($this->root, $this->pathInfo($path, 'path'));
        if ($target instanceof FileInterface) {
            $target->detach();
            unset($target);
            return true;
        }
        return false;
    }


    public function write($path, $contents, Config $config = null)
    {
        $file = $this->makeFile($this->pathInfo($path, 'path'), $this->root);
        $file->writeContent($contents);

        $type = $file->getType();
        $size = $file->getSize();
        $contents = $file->getContent();
        $result = compact('contents', 'type', 'size', 'path');

        if ($config && $visibility = $config->get('visibility')) {
            $result['visibility'] = $visibility;
            $this->setVisibility($path, $visibility);
        }
        return $result;
    }

    public function writeStream($path, $resource, Config $config = null)
    {
        $file = $this->makeFile($this->pathInfo($path, 'path'), $this->root);
        $file->writeStream($resource);

        $type = $file->getType();
        $size = $file->getSize();
        $contents = $file->getContent();
        $result = compact('contents', 'type', 'size', 'path');

        if ($config && $visibility = $config->get('visibility')) {
            $result['visibility'] = $visibility;
            $this->setVisibility($path, $visibility);
        }
        return $result;
    }

    public function update($path, $contents, Config $config = null)
    {
        $file = $this->makeFile($this->pathInfo($path, 'path'), $this->root);
        $file->updateContent($contents);

        $type = $file->getType();
        $size = $file->getSize();
        $result = compact('contents', 'type', 'size', 'path');

        if ($config && $visibility = $config->get('visibility')) {
            $result['visibility'] = $visibility;
            $this->setVisibility($path, $visibility);
        }
        return $result;
    }

    public function updateStream($path, $resource, Config $config = null)
    {
        $file = $this->makeFile($this->pathInfo($path, 'path'), $this->root);
        $file->updateStream($resource);

        $type = $file->getType();
        $size = $file->getSize();
        $contents = $file->getContent();
        $result = compact('contents', 'type', 'size', 'path');

        if ($config && $visibility = $config->get('visibility')) {
            $result['visibility'] = $visibility;
            $this->setVisibility($path, $visibility);
        }
        return $result;
    }

    public function copy($from, $to): bool
    {
        $target = $this->get($this->root, $this->pathInfo($from, 'path'));
        if (!$target || $this->has($to)) {
            return false;
        }

        if ($target instanceof DirInterface) {
            foreach ($target->listContents() as $content) {
                $this->copy($content->getFilePath(), $to . '/' . $content->getFileName());
            }
        } else {
            $newPath = $this->pathInfo($to, 'path');
            $fileName = array_pop($newPath);
            $file = new File($fileName);
            $file->writeContent($target->getContent());
            $file->attach($this->makeDir($newPath, $this->root));
        }
        return true;
    }

    public function rename($path, $newpath): bool
    {
        $target = $this->get($this->root, $this->pathInfo($path, 'path'));
        if (!$target || $this->has($newpath)) {
            return false;
        }

        $newPath = $this->pathInfo($newpath, 'path');
        $fileName = array_pop($newPath);
        $target->rename($fileName);
        $target->attach($this->makeDir($newPath, $this->root));
        return true;
    }

    public function createDir($dirname, Config $config = null)
    {
        $dir = $this->makeDir($this->pathInfo($dirname, 'path'), $this->root);
        return ['path' => $dir->getFilePath(), 'type' => $dir->getType()];
    }

    public function deleteDir($dirname)
    {
        $target = $this->get($this->root, $this->pathInfo($dirname, 'path'));
        if ($target instanceof DirInterface) {
            $target->detach();
            unset($dir);
            return true;
        }
        return false;
    }

    public function listContents($directory = '', $recursive = false)
    {
        $target = $this->get($this->root, $this->pathInfo($directory, 'path'));
        if ($target instanceof DirInterface) {
            return array_map([$this, 'normalizeFileInfo'], $target->listContents());
        }
        return [];
    }

    public function getMetadata($path)
    {
        $target = $this->get($this->root, $this->pathInfo($path, 'path'));
        return $target ? $this->normalizeFileInfo($target) : false;
    }

    public function getSize($path)
    {
        $target = $this->get($this->root, $this->pathInfo($path, 'path'));
        return $target ? $target->getSize() : false;
    }

    public function getMimetype($path)
    {
        $target = $this->get($this->root, $this->pathInfo($path, 'path'));
        if ($target instanceof FileInterface) {
            return ['path' => $path, 'type' => 'file', 'mimetype' => 'text/plain'];
        }
        return false;
    }

    public function getTimestamp($path)
    {
        $target = $this->get($this->root, $this->pathInfo($path, 'path'));
        return $target ? $target->getTimestamp() : false;
    }

    protected function normalizeFileInfo($file): array
    {
        $normalized = [
            'type' => $file->getType(),
            'path' => $file->getFilePath(),
            'timestamp' => $file->getTimestamp(),
        ];
        if ($normalized['type'] === 'file') {
            $normalized['size'] = $file->getSize();
        }
        return $normalized;
    }
}
