<?php

namespace YiluTech\FileCenter\Adapter;

use League\Flysystem\FileExistsException;

abstract class BaseFileAbstract
{
    protected string $name;

    protected int $lastModified;

    protected ?DirInterface $target;

    protected $visibility;

    public function __construct(string $name)
    {
        if ($name == '.' || $name == '..') {
            throw new FileExistsException($name);
        }
        $this->name = $name;
        $this->lastModified = time();
    }

    public function getFileName(): string
    {
        return $this->name;
    }

    public function getDirPath(): string
    {
        return isset($this->target) ? $this->target->getFilePath() : '';
    }

    public function getFilePath(): string
    {
        return isset($this->target) ? $this->target->getFilePath() . '/' . $this->getFileName() : $this->getFileName();
    }

    public function getTimestamp(): int
    {
        return $this->lastModified;
    }

    public function attach(DirInterface $target)
    {
        $this->detach();
        $target->addFile($this);
        $this->target = $target;
    }

    public function detach()
    {
        if (isset($this->target)) {
            $this->target->removeFile($this);
            $this->target = null;
        }
    }

    public function rename(string $name)
    {
        if ($this->name === $name) {
            return;
        }
        if ($name === '.' || $name === '..') {
            throw new FileExistsException($this->getDirPath() . '/' . $name);
        }
        if (empty($this->target)) {
            $this->name = $name;
            return;
        }

        $target = $this->target;
        if ($exists = $target->getFile($name)) {
            throw new FileExistsException($exists->getFilePath());
        }
        $this->detach();
        $this->name = $name;
        $this->attach($target);
    }

    protected function updateLastModify()
    {
        $this->lastModified = time();
    }

    public function setVisibility($visibility)
    {
        $this->visibility = $visibility;
    }

    public function getVisibility()
    {
        return $this->visibility;
    }
}
